#ifndef __DM_CRESNET_SLAVE_H__
#define __DM_CRESNET_SLAVE_H__

/**
 *
 * \file        dm_cresnet_slave.h
 *
 * \brief       Interface for Crestron's proprietary protocol for device
 *              communication over serial or IP.
 *
 * \author      Pete McCormick (copied from network.h from Cajita code)
 *
 * \date        12/5/2007
 *
 */

////////////////////////////////////////////////////////////////////////////////

#include "dm_nutypedefs.h"
#include "dm_os.h"

#define CNET_PKT_HEADER               2
#define CNET_DIGITAL_SIZE_OFFSET      2
#define CNET_ANALOG_CLX_SIZE_OFFSET   2
#define CNET_ANALOG_SIZE_OFFSET       4
#define CNET_ANALOG_RCB_SIZE_OFFSET   5

#define DM_DEFAULT_CNET_SLAVE_QUEUE_SIZE        40

#define DM_CRESNET_SLAVE_SEND_PENDING_TIMER_MS  9 // 20

#ifdef __cplusplus
extern "C" {
#endif

// Start and end message flags for serial
#define START_NOT_END  1
#define APPEND_NOT_END 0
#define APPEND_AND_END 2
#define START_AND_END  3

typedef void (*CNET_CUSTOM)(void *packet);

typedef void (*CNET_ENDOFUPDATE)(UINT8 bStream, UINT8 source);
typedef void (*PCNET_PACKET2C)(UINT8 *packet);

// Type of function pointer: What kind of function the CrestCore library
//  calls to handle each received Data Transfer "Block Complete" message.
typedef void (*CNET_XFER_BLOCK_COMPLETE_FTYPE)(UINT8 bStream);

// ...How an application sets the function pointer that the CrestCore library
//  calls to handle each received Data Transfer "Block Complete" message.
extern void SetFuncForCnetXferBlockComplete (CNET_XFER_BLOCK_COMPLETE_FTYPE funcPtr);

////////////////////////////////////////////////////////////////////////////////

typedef struct
{
    UINT8 id;
    UINT32 PPNAddress;
    unsigned int selectiveAllClear;
    BOOL disableIso8859;
} CRESNETPARAMS;
////////////////////////////////////////////////////////////////////////////////
// Structures for memorizing joins and their state changes
struct DIGITAL_JOIN_STATE
{
    UINT32 JoinNumber:24;   // Bits 0-15: Join value (max = 32767)
                            // Bits 16-23: Sub slot #
    UINT8 CurrentValue:1;   // latest value of join
    UINT8 Changed:1;        // if true join has changed but not sent to switch
    UINT8 Source:2;         // source of message (also where to send response)
    UINT8 ChangedRTJI:1;    // if true join has changed but not sent to RTJI
    UINT8 JoinDefined:1;        // if true join has been JoinDefined
    UINT8 Reserved:2;       // Reserved bits to align to a byte
};

//  bits defining the analog packet type stored in the ANALOG_JOIN_STATE
#define ANALOG_PKT_TYPE_RCB 1
#define ANALOG_PKT_TYPE_CLX 2

struct ANALOG_JOIN_STATE
{
    UINT32 JoinNumber:24;   // Bits 0-15: Join value (max = 32767)
                            // Bits 16-23: Sub slot #
    UINT8 Changed:1;        // if true join has changed but not sent to switch
    UINT8 Source:2;         // source of message (also where to send response)
    UINT8 ChangedRTJI:1;    // if true join has changed but not sent to RTJI
    UINT8 Reserved:1;       // Reserved bits to align to a byte
    UINT8 JoinDefined:1;    // if true join has been Join Defined
    UINT8 JoinPType:2;      // if 0=send as analog, ANALOG_PKT_TYPE_RCB=send as rcb, ANALOG_PKT_TYPE_CLX=send as clx

    UINT16 CurrentValue;

#ifdef  RCB_ANALOG_SUPPORT
    UINT8  Flags;           // RCB packet flags
    UINT32 Time;            // RCB packet time
#endif

} ;

struct SERIAL_JOIN_STATE
{
    UINT32 JoinNumber:24;   // Bits 0-15: Join value (max = 32767)
                            // Bits 16-23: Sub slot #
    UINT8 Changed:1;        // if true join has changed but not sent to switch
    UINT8 Source:2;         // source of message (also where to send response)
    UINT8 ChangedRTJI:1;    // if true join has changed but not sent to RTJI
    UINT8 Reserved:3;       // Reserved bits to align to a byte
    UINT8 JoinDefined:1;    // if true join has been JoinDefined
};

#define DM_WINCE_SERIAL_MAX_TEXT_SIZE			1024

struct SERIAL_JOIN_STATE_TEXT
{
	UINT8 LastFlags;
	UINT8 sText[DM_WINCE_SERIAL_MAX_TEXT_SIZE];
	UINT16 sCurrentLength;
};

struct COMMAND_RESPONSE
{
    UINT8 Command;       // cresnet command
    UINT8 Type;          // sub type of command
    UINT8 Transaction;   // transaction number
    UINT8 Source;        // source of message
    UINT8 *pPacket;      // pointer to the input packet, if needed
    UINT8 slot;          // slot
};

class CDMCresnetUartHost;

class CresnetJoinState
{
public:

#ifdef DM_WINCE
  CresnetJoinState(UINT8 bStream, UINT16 NumDigitals, UINT16 NumAnalogs, UINT16 NumSerials, UINT8 NumCommands, UINT8 bPackJoinLimit = 0, UINT8 bSaveSerialText = 0);
#else
  CresnetJoinState(UINT8 bStream, UINT16 NumDigitals, UINT16 NumAnalogs, UINT16 NumSerials, UINT8 NumCommands, UINT8 bPackJoinLimit = 0);
#endif

  ~CresnetJoinState();
  void CresnetDigitalMemorize(UINT32 join, BOOL on, UINT8 bSource, UINT8 bForce = false, UINT8 bDoNotSend = false);
  void CresnetAnalogMemorize(UINT32 join, UINT16 level, UINT8 bSource, UINT8 bForce = false, UINT32 rcbclxtime = 0, UINT8 rcbflags = 0, UINT8 pkttype = 0, UINT8 bDoNotSend = false);

#ifdef DM_WINCE
  void CresnetSerialMemorize(UINT32 join, UINT8 bSource, UINT8* pPacket = NULL);  
#else
  void CresnetSerialMemorize(UINT32 join, UINT8 bSource);
#endif

  BOOL CresnetDigitalRecall( UINT32 sJoin );
  UINT16 CresnetAnalogRecall( UINT32 join );
  UINT8 * CresnetSerialRecall(UINT32 join, UINT32 * pByteCnt);
  void CresnetCommandMemorize(UINT8 *bRequest, UINT8 bSource, UINT8 bSlot);
  UINT8 CresnetNextDigitalPending(void);
  UINT8 CresnetNextAnalogPending(void);
  UINT8 CresnetNextSerialPending(void);
  UINT8 CresnetNextCommandPending(void);
  void ProcessRequestTJI( unsigned char* packet, UINT8 bCresnet, UINT8 bSlot);

#ifdef DM_WINCE
  //WinCE specific functions that will memorize the serial string text
  UINT8* GetLocalSerialString(UINT16 sIndex, UINT16 *lLength);
  void SaveLocalSerialString(UINT16 sIndex, UINT8* pPacket);
#endif

  //Processes the all clear
  void ProcessAllClear (UINT8 bForce = false, UINT8 bReportSerial = false);

  //Functions to return current status
  UINT16 GetMaxDigitalCount( void );
  UINT16 GetMaxAnalogCount( void );
  UINT16 GetMaxSerialCount( void );

  UINT16 GetUsedDigitalCount( void );
  UINT16 GetUsedAnalogCount( void );
  UINT16 GetUsedSerialCount( void );

  void SetMaxOutputBuffer( UINT8 MaxOutput );

  void StartCommandRequestAckTimer( UINT8 bSource );
  void CheckCommandRequestAckTimer( void );

  void SetJoinPackLimit( UINT8 bLimit )
  {
      m_bPackJoinLimit = bLimit;
  }

  UINT8 GetJoinPackLimit( void )
  {
      return( m_bPackJoinLimit );
  }

private:
    INT32 m_iCommandRequestAckTimer;
    UINT8 m_bCommandRequestAckSource;

    UINT8 dig_val_in(UINT32 sJoin, UINT8* pStatus);
    UINT16 get_analog_in_value(UINT32 sJoin, UINT8* pStatus);

    UINT8 *GetOutputBuffer( UINT8 *MaxOutput );

    UINT8 m_Stream;
    UINT16 m_NumDigitals;      // max number of digitals
    UINT16 m_NumAnalogs;      // max number of analog (16 bit)
    UINT16 m_NumSerials;
    UINT8 m_NumCommands;
    DIGITAL_JOIN_STATE *m_pDigital; // pointer to digital states
    ANALOG_JOIN_STATE *m_pAnalog;   // pointer to analog states
    SERIAL_JOIN_STATE *m_pSerial;   // pointer to serial states
    COMMAND_RESPONSE *m_pCommands;  // pointer to command responses

    UINT16 m_bDigitalJoinCount;
    UINT16 m_bAnalogJoinCount;
    UINT16 m_bSerialJoinCount;

    UINT8 MaxOutputLength;
    UINT8 m_bPackJoinLimit;

#ifdef DM_WINCE
	UINT8 m_bSaveSerialText;
	SERIAL_JOIN_STATE_TEXT* m_pSerialText;
#endif
};

typedef struct
{
    BOOL debug;
    BOOL disableIso8859;
    UINT8 dest;
    UINT8 NumStreams;
    UINT32 inQueue;
    UINT8 extra[OS_TASK_EXTRA_BYTES];
    UINT32 allClearCnt;
    UINT32 blockBytes;
    CresnetJoinState *pCresnetJoinState;
}CRESNETSLAVE;

enum
{
  COMMAND_CHECK,
  DIGITAL_CHECK,
  ANALOG_CHECK,
  SERIAL_CHECK,
  DM_HOST_CHECK,
  MAX_MESSAGE_TYPES
};

// Replaced emun list that implies a variable list. These are fixed numbers that are permanently burned into s-record files - values will never change.
#define    FLASH_FPGA           0  // download image is for the FPGA
#define    FLASH_DSP            1  // download image is for the DSP
#define    FLASH_BOOTLOADER     2  // download image is for the Bootloader
#define    FLASH_OMRON_RX       3  // download image is for the Omron Fiber RX module
#define    FLASH_OMRON_TX       4  // download image is for the Omron Fiber TX module
#define    FLASH_VALENS_RX      5  // download image is for remote Valens VS100 RX
#define    FLASH_VALENS_TX      6  // download image is for remote Valens VS100 RX
#define    FLASH_GEN_SCALER     7  // download image is Genum scaler
#define    FLASH_SECOND_FPGA    8  // download image is for second FPGA
#define    FLASH_OSD_SPI        9
#define    FLASH_MARS          0x0A  // download image is for MARS (or Genesis)
#define    FLASH_SECOND_DSP    0x0B  // download image is for second DSP
#define    FLASH_PREBOOTLOADER 0x0C  // download image is for pre-bootloader
#define    FLASH_MG3500        0x0D  // download image is for MG3500
#define    MAX_FLASH_TYPES     0x0E


// This enum must match the bootloader list here: http://fwsql-10/devicedb/Bootloaders_list.asp
enum
{
    BOOTLOAD_STM32_128,     // 0 bootloader for STM32 processors with 128K memory
    BOOTLOAD_STM32_256,     // 1 bootloader for STM32 processors with 256K memory
    BOOTLOAD_STR9,          // 2 bootloader for standard STR9 processor configurations
    BOOTLOAD_STR9B,         // 3 bootloader for alternate (e.g. roombox) STR9 processor configurations
    BOOTLOAD_STR9MP,        // 4 bootloader for MP Series STR9 processor (full cresnet addressing)
    BOOTLOAD_RM,            // 5 bootloader for TX's (ethernet upload, Reverse cresnet)
    BOOTLOAD_IR,            // 6 bootloader for IR controller (i2c communications)
    BOOTLOAD_TX,            // 7 bootloader for STM32 processor TX (DM-TX1-1G)
    BOOTLOAD_STR9DGEMX,     // 8 bootloader for DGE-2
    BOOTLOAD_STR9PSAHD,     // 9 bootloader for PSAHD
    BOOTLOAD_STM32B,        // 10 bootloader for 101CB processor
    BOOTLOAD_SWAMP_E,       // 11 bootloader for SWAMP-E
    BOOTLOAD_STR9DINDALI2,  // 12 bootloader for DINDALI2
    BOOTLOAD_STM32_1024,    // 13 bootloader for STM32 processors with 1024K memory
    BOOTLOAD_STR9GLS_EM,    // 14 bootloader for GLS-EM-MCU
    BOOTLOAD_STM32_USBCNET, // 15 bootloader for STM32 1M memory, cresnet over USB
    BOOTLOAD_HDXSP,         // 16 bootloader for HDXSP
    BOOTLOAD_FPTOUCH,       // 17 bootloader for FPTOUCH
    BOOTLOAD_TSI,           // 18 bootloader for TSx IR controller (I2C communications)
    BOOTLOAD_SWPOE16,       // 19 bootloader for POE 16
    BOOTLOAD_DMF2,          // 20 bootloader for DM STM32F2
    BOOTLOAD_C2NIO,         // 21 bootloader for C2N-IO
    BOOTLOAD_IOEX,          // 22 bootloader for IOEX
    BOOTLOAD_FLIPTOP,       // 23 bootloader for flip top series
    BOOTLOAD_TABLETOP,        // 24 bootloader for ConnectIt TableTop Presentation Interface
	BOOTLOAD_STR9USB,       // 25 bootloader for STR9 processor with USB only configurations (no Ethernet)
    MAX_DM_BOOTLOADERS
};

enum
{
  PREBOOTLOADER_STM32F2,        // pre-bootloader for DM STM32F2
  MAX_DM_PREBOOTLOADERS
};
// flag to compare image instead of programming it
#define FLASH_VERIFY 0x80

// Pointer to array of Join state objects - one for each stream
extern CresnetJoinState **pJoinState;
extern const UINT8 EncapOnlinePacket[];
extern const UINT8 EncapOfflinePacket[];

#define PACKET_LENGTH_IDX 1
#define RCON_FLAG_IDX 3
#define RCON_DATA_IDX 4

#define PASSTO_FLAG_IDX 3
#define PASSTO_DATA_IDX 4

////////////////////////////////////////////////////////////////////////////////

//extern short CresnetTxCount;
//extern BOOL CNetRedirectActive;

////////////////////////////////////////////////////////////////////////////////

extern UINT32 CresnetSlaveRemapSubslottedJoinToStream( UINT32 lJoin, UINT8* pStream );
extern void CresnetSlaveProcessInternalDigital(UINT8 bStream, UINT32 join, UINT32 value, UINT32 source);
extern void CresnetSlaveProcessInternalAnalog(UINT8 bStream, UINT32 join_number, UINT32 value, UINT32 source);
extern void CresnetSlaveProcessInternalSerial(UINT8 bStream, UINT32 join, char * value, UINT32 length, UINT32 source,UINT8 flag = START_AND_END);

extern void CresnetSlaveProcessInternalRCB(UINT8 bStream, UINT32 join, UINT32 analog_value, UINT32 time, UINT8 flag, UINT32 source);
extern void CresnetSlaveProcessInternalCLX(UINT8 bStream, UINT32 join, UINT32 analog_value, UINT32 time, UINT32 source);
extern void CresnetSlaveProcessIr( UINT8 *packet, UINT8 packlen, UINT32 source);
void CresnetSlaveProcessInternalCommand( UINT8 *packet, UINT8 packlen, UINT32 source);
extern UINT32 CresnetSlaveGetPPNAddr(void);
extern void CresnetSlaveSetPPNAddr(UINT32);
//extern int CresnetValidateLocalMsg(UINT8 *packet);
extern BOOL CresnetSlaveRxParse(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot);
extern void CresnetSlaveRxParseCmd(UINT8 *packet, UINT32 source, UINT32 maxlen, UINT32 slot);
//extern void CresnetAcceptIncomingPacket(UINT8 *packet, UINT32 source);
//extern void CresnetEnqueueOutgoingPacket(void *packet, UINT32 packet_size, UINT32 destination);
extern BOOL CresnetSlaveSendType29( UINT8 bStream, UINT32 slot, UINT16 time, UINT16 delta, UINT32 dest );
extern BOOL CresnetSlaveSendType2B( UINT8 bStream, UINT16 ch, UINT16 temp, UINT32 dest );
extern BOOL CresnetSlaveSendType09( UINT8 bStream, UINT32 join, UINT8 data0, UINT8 data1, UINT8 data2, UINT32 dest );
extern int CresnetSlaveSendDigital(UINT8 bStream, UINT32 numb, BOOL on, UINT32 destination);
extern int CresnetSlaveSendDigitalRepeat( UINT8 bStream, UINT32 join, BOOL on, UINT32 dest );
extern int CresnetSlaveSendAnalog(UINT8 bStream, UINT32 transmit_channel, UINT16 level, UINT32 destination);
extern int CresnetSlaveSendSerial(UINT8 bStream, UINT8 *message,
                         UINT32 message_length, UINT32 message_number, UINT32 destination);
void CresnetSlaveDebugEnable(UINT8 bEnable);
void CresnetParamsSave(UINT8 CNetID);
extern void CresnetParamsRestore(void);
void CresnetSlaveProcessAllClear(UINT8 bStreamId, UINT8 source);
void CresnetSlaveSendReadyForXfer(UINT32 blockBytes, UINT8 bStream);
UINT8 CresnetGetDefaultId(void);
extern UINT8 GetStreamFromPacket( UINT8 bAddr );
extern UINT8 GetPacketFromStream( UINT8 bStream );
void SendOnlineStatusToController(UINT8 bStream, bool online);
// Are there any libraries specific header files
extern void SendDMLinkStatusToController(UINT8);


#define ALL_STREAMS 0xff


void CresnetSlaveEnqueueOutgoingPacket(UINT8 bStream, void *packet, UINT32 packet_size,
                                       UINT32 destination);

UINT8 IsCrestnetSlaveQueueFull(UINT8 bStream, UINT8 bDestination);
CresnetJoinState *GetState(UINT8 bStream);
void CresnetSlaveSendPending(void);
int CresnetSlaveSendCapabilities(UINT8 bStream, COMMAND_RESPONSE *bRequest);
int CresnetSlaveSendExtendedCapabilities(UINT8 bStream, COMMAND_RESPONSE *bRequest);
UINT8 *GetCresnetSerialString(UINT8 bStream, UINT32 join, UINT16 *lLength);

extern void PowerLED(UINT8 on);
extern void CresnetLED(UINT8 on);
extern void PPNSetupLED(UINT8 on);
extern void HeartBeatLED(UINT8 on);
extern UINT32 CresnetSlaveAcceptIncomingPacket(UINT8 *packet);

//Allow packed joins
extern void CresnetSlaveInit( UINT8 NumStream, UINT16 NumDigitals, UINT16 NumAnalogs, UINT16 NumSerials, 
                              UINT8 NumCommands, UINT32 blockSize = 0, UINT8 bPackJoinLimit = 0, 
                              UINT8 bQueueSize = DM_DEFAULT_CNET_SLAVE_QUEUE_SIZE, 
                              UINT16 additionalstacksize = 0, BOOL CombinedPackets = FALSE, 
                              UINT8 bRconTimerBufferCount = 0 );

extern void CresnetSlaveSetUploadBlockSize(UINT16 blockSize);
extern void CresnetDataXferData(UINT32 addr, UINT8 * pData, UINT8 byteCnt,
                                BOOL csumValid, UINT8 bStream = 0);
extern BOOL CresnetDataXferStart(UINT32 flashId, UINT32 *pBlockSize = 0, UINT8 bStream = 0, UINT32 *pCodeSize = 0);
extern void CresnetDataXferEnd(UINT8 bStream = 0);
extern void CresnetDataXferAbort(UINT8 bStream = 0);
extern void * FindMasterByDeviceID(UINT32 destination);
extern void CresnetProcessTransferStart( void );
extern void CresnetProcessTransferEnd( UINT8 bTransferAborted );
extern void WriteBootloaderVectorBlockIfNeeded(void);
void CresnetBootloaderRequest(BOOL doBootloader, UINT8 bStream);

CresnetJoinState *GetCresnetJoinStatePointer(INT8 bStream)  ;
UINT32 ApplySubSlotMaskToJoinNumber( UINT8 slot, UINT32 join );

//Temporary for switch code base
void UartCresnetSlaveSetAutoId(UINT8 bStream);

void ClearBootloaderSRecordData(void);

void SetCnetCustomFunctionPointer( CNET_CUSTOM pfCnetCustom );


typedef struct
{
    UINT32 writeAddr;
    UINT32 xferErrCnt;
    UINT32 xferCsumErrCnt;
    UINT32 xferAddrErrCnt;
    UINT32 xferWriteErrCnt;
    UINT32 xferVerifyErrCnt;
    UINT32 xferBytes;
    UINT16 crc16;
    UINT8 flashId;
    UINT8 PromCompare;
}CNETHELPER;
void WriteBootloaderFlashRecord(UINT32 addr, UINT8 * pData, UINT8 byteCnt, CNETHELPER *CnetHelper);
void CresnetFlashLoaderSendAbortXfer(void);

#define UPLOAD_BLOCK_SIZE 512

extern UINT8 GetNumStreams( void );

// make end of update a function pointer, this way not all projects have to have it
#define CLEAR_ALL_STATE 0x80
#define CHECK_END_OF_UPDATE_TIMEOUT 0x40

extern CNET_ENDOFUPDATE pFnEndOfUpdate;
extern void CresnetSlaveProcessEndOfUpdate (UINT8 program, UINT8 source);

extern void CresnetSlaveProcessWinCeEndOfUpdate (UINT8 bStream);

// For rerouting cresnet packets before/after the main parsing
extern BOOL (*pfPreCnetSlaveParsePktIntercept)(UINT8* pPkt, UINT8 PktLen, UINT8 PktType, UINT8 Source);
extern void (*pfPostCnetSlaveParsePktProcess)(UINT8* pPkt, UINT8 PktLen, UINT8 PktType, UINT8 Source);
////////////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
}
#endif

#endif  //__DM_CRESNET_SLAVE_H__

